'use client'

import { useState, useEffect } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Users, GraduationCap, BookOpen, Layers, LogOut, Home } from 'lucide-react'
import { useToast } from '@/hooks/use-toast'

interface DashboardProps {
  onBack: () => void
}

export default function AdminDashboard({ onBack }: DashboardProps) {
  const { toast } = useToast()
  const [stats, setStats] = useState({ guru: 0, siswa: 0, kelas: 0, mapel: 0 })
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    fetchStats()
  }, [])

  const fetchStats = async () => {
    try {
      const response = await fetch('/api/stats')
      const data = await response.json()
      if (data.success) {
        setStats(data.stats)
      }
    } catch (error) {
      toast({
        title: 'Error',
        description: 'Gagal mengambil statistik',
        variant: 'destructive'
      })
    } finally {
      setLoading(false)
    }
  }

  const handleLogout = async () => {
    try {
      await fetch('/api/auth/logout', { method: 'POST' })
      toast({
        title: 'Berhasil',
        description: 'Logout berhasil'
      })
      onBack()
    } catch (error) {
      toast({
        title: 'Error',
        description: 'Gagal logout',
        variant: 'destructive'
      })
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-green-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8 px-4">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="flex justify-between items-center mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-800">Dashboard Admin</h1>
            <p className="text-gray-600">Selamat datang di panel admin</p>
          </div>
          <div className="flex gap-2">
            <Button onClick={onBack} variant="outline">
              <Home className="mr-2 h-4 w-4" />
              Back to Home
            </Button>
            <Button onClick={handleLogout} variant="destructive">
              <LogOut className="mr-2 h-4 w-4" />
              Logout
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <StatCard
            icon={<Users className="h-8 w-8" />}
            title="Jumlah Guru"
            value={stats.guru}
            color="from-yellow-400 to-yellow-500"
          />
          <StatCard
            icon={<GraduationCap className="h-8 w-8" />}
            title="Jumlah Siswa"
            value={stats.siswa}
            color="from-green-400 to-green-500"
          />
          <StatCard
            icon={<Layers className="h-8 w-8" />}
            title="Jumlah Kelas"
            value={stats.kelas}
            color="from-yellow-400 to-yellow-500"
          />
          <StatCard
            icon={<BookOpen className="h-8 w-8" />}
            title="Mata Pelajaran"
            value={stats.mapel}
            color="from-green-400 to-green-500"
          />
        </div>

        {/* Menu Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <MenuCard
            title="Data Guru"
            description="Kelola data guru - tambah, edit, hapus"
            icon={<Users className="h-12 w-12" />}
            color="bg-gradient-to-br from-yellow-400 to-yellow-500"
          />
          <MenuCard
            title="Data Siswa"
            description="Kelola data siswa, upload Excel, cetak PDF"
            icon={<GraduationCap className="h-12 w-12" />}
            color="bg-gradient-to-br from-green-400 to-green-500"
          />
          <MenuCard
            title="Data Kelas"
            description="Kelola data kelas"
            icon={<Layers className="h-12 w-12" />}
            color="bg-gradient-to-br from-yellow-400 to-yellow-500"
          />
          <MenuCard
            title="Mata Pelajaran"
            description="Kelola mata pelajaran"
            icon={<BookOpen className="h-12 w-12" />}
            color="bg-gradient-to-br from-green-400 to-green-500"
          />
        </div>
      </div>
    </div>
  )
}

function StatCard({ icon, title, value, color }: any) {
  return (
    <Card className="shadow-lg hover:shadow-xl transition-all duration-300">
      <CardContent className="p-6">
        <div className={`bg-gradient-to-r ${color} rounded-lg p-4 w-16 h-16 flex items-center justify-center text-white mb-4`}>
          {icon}
        </div>
        <p className="text-gray-600 text-sm mb-1">{title}</p>
        <p className="text-3xl font-bold text-gray-800">{value}</p>
      </CardContent>
    </Card>
  )
}

function MenuCard({ title, description, icon, color }: any) {
  return (
    <Card className="shadow-lg hover:shadow-2xl transition-all duration-300 cursor-pointer group">
      <CardContent className="p-6">
        <div className={`${color} rounded-lg p-4 w-16 h-16 flex items-center justify-center text-white mb-4 group-hover:scale-110 transition-transform duration-300`}>
          {icon}
        </div>
        <h3 className="text-xl font-bold text-gray-800 mb-2">{title}</h3>
        <p className="text-gray-600 text-sm">{description}</p>
      </CardContent>
    </Card>
  )
}
