import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

// GET - Ambil semua data siswa dengan pagination dan search
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const search = searchParams.get('search') || ''
    const page = parseInt(searchParams.get('page') || '1')
    const limit = parseInt(searchParams.get('limit') || '10')
    const sortBy = searchParams.get('sortBy') || 'createdAt'
    const sortOrder = searchParams.get('sortOrder') || 'desc'
    const skip = (page - 1) * limit

    // Build where clause for search
    const where = search
      ? {
          OR: [
            { nama: { contains: search } },
            { nisn: { contains: search } },
            { induk: { contains: search } },
            { kelas: { contains: search } }
          ]
        }
      : {}

    // Get total count for pagination
    const total = await db.siswa.count({ where })

    // Get siswa data with pagination
    const siswaList = await db.siswa.findMany({
      where,
      skip,
      take: limit,
      orderBy: {
        [sortBy]: sortOrder
      }
    })

    return NextResponse.json({
      success: true,
      data: siswaList,
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit)
      }
    })
  } catch (error) {
    console.error('Get siswa error:', error)
    return NextResponse.json(
      { error: 'Terjadi kesalahan saat mengambil data siswa' },
      { status: 500 }
    )
  }
}

// POST - Tambah siswa baru
export async function POST(request: NextRequest) {
  try {
    const body = await request.json()

    // Validasi field wajib
    const requiredFields = ['induk', 'nisn', 'nama', 'tanggalLahir', 'kelas']
    const missingFields = requiredFields.filter(field => !body[field])

    if (missingFields.length > 0) {
      return NextResponse.json(
        {
          error: `Field berikut harus diisi: ${missingFields.join(', ')}`
        },
        { status: 400 }
      )
    }

    const siswa = await db.siswa.create({
      data: body
    })

    return NextResponse.json({ success: true, data: siswa }, { status: 201 })
  } catch (error) {
    console.error('Create siswa error:', error)
    return NextResponse.json(
      { error: 'Terjadi kesalahan saat menambah siswa' },
      { status: 500 }
    )
  }
}
